<?php
/**
 * Debug Credit Reminders
 * This script specifically tests the Send Reminder functionality from credit.php
 * It simulates the exact same process that happens when you click "Send Reminder"
 */

session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once 'api_helpers.php';
require_once '../includes/functions.php';
require_once '../includes/sms_functions.php';
require_once '../includes/email_functions.php';
require_once '../includes/whatsapp_functions.php';
require_once '../includes/email_templates.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    echo "Please log in first to test reminder functionality.";
    exit;
}

?>
<!DOCTYPE html>
<html>
<head>
    <title>Debug Credit Reminders</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .success { background: #c8e6c9; padding: 10px; margin: 10px 0; border-radius: 5px; }
        .error { background: #ffcdd2; padding: 10px; margin: 10px 0; border-radius: 5px; }
        .warning { background: #fff3e0; padding: 10px; margin: 10px 0; border-radius: 5px; }
        .info { background: #e3f2fd; padding: 10px; margin: 10px 0; border-radius: 5px; }
        .test-button { background: #2196F3; color: white; padding: 10px 20px; border: none; border-radius: 5px; cursor: pointer; margin: 5px; }
        .test-button:hover { background: #1976D2; }
        pre { background: #f5f5f5; padding: 10px; border-radius: 5px; overflow-x: auto; }
        #results { margin-top: 20px; }
    </style>
</head>
<body>

<h1>🔍 Debug Credit Reminders</h1>
<p>This tool tests the exact same functionality that happens when you click "Send Reminder" in credit.php</p>

<?php
// Get a test customer with unpaid balance
try {
    $stmt = $pdo->prepare("
        SELECT c.id, c.name, c.phone, c.email,
               SUM(ct.amount - COALESCE(cp_summary.total_paid, 0) - COALESCE(cp_summary.total_waived, 0)) as balance
        FROM customers c
        LEFT JOIN credit_transactions ct ON c.id = ct.customer_id
        LEFT JOIN (
            SELECT 
                credit_transaction_id,
                SUM(CASE WHEN is_waived = 0 OR is_waived IS NULL THEN amount ELSE 0 END) as total_paid,
                SUM(CASE WHEN is_waived = 1 THEN amount ELSE 0 END) as total_waived
            FROM credit_payments
            GROUP BY credit_transaction_id
        ) cp_summary ON ct.id = cp_summary.credit_transaction_id
        WHERE ct.id IS NOT NULL
        GROUP BY c.id, c.name, c.phone, c.email
        HAVING balance > 0.001
        ORDER BY balance DESC
        LIMIT 5
    ");
    $stmt->execute();
    $testCustomers = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (empty($testCustomers)) {
        echo "<div class='warning'>❌ No customers with unpaid balances found for testing.</div>";
        exit;
    }

    echo "<h2>🧪 Available Test Customers:</h2>";
    echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
    echo "<tr><th>ID</th><th>Name</th><th>Phone</th><th>Email</th><th>Balance</th><th>Actions</th></tr>";
    
    foreach ($testCustomers as $customer) {
        echo "<tr>";
        echo "<td>" . htmlspecialchars($customer['id']) . "</td>";
        echo "<td>" . htmlspecialchars($customer['name']) . "</td>";
        echo "<td>" . htmlspecialchars($customer['phone'] ?: 'None') . "</td>";
        echo "<td>" . htmlspecialchars($customer['email'] ?: 'None') . "</td>";
        echo "<td>Ksh " . number_format($customer['balance'], 2) . "</td>";
        echo "<td>";
        echo "<button class='test-button' onclick='testReminder(" . $customer['id'] . ", \"sms\")'>Test SMS</button>";
        echo "<button class='test-button' onclick='testReminder(" . $customer['id'] . ", \"email\")'>Test Email</button>";
        echo "<button class='test-button' onclick='testReminder(" . $customer['id'] . ", \"whatsapp\")'>Test WhatsApp</button>";
        echo "</td>";
        echo "</tr>";
    }
    echo "</table>";

} catch (Exception $e) {
    echo "<div class='error'>❌ Database Error: " . htmlspecialchars($e->getMessage()) . "</div>";
    exit;
}
?>

<h2>📧 Test Results:</h2>
<div id="results">
    <p><em>Click one of the test buttons above to test reminder functionality...</em></p>
</div>

<h2>🔧 Configuration Status:</h2>
<?php
// Check configurations
$stmt = $pdo->prepare("SELECT setting_group, setting_key, setting_value FROM store_settings WHERE setting_group IN ('sms_settings', 'email_settings', 'whatsapp_settings')");
$stmt->execute();
$allSettings = $stmt->fetchAll(PDO::FETCH_ASSOC);

$settingsByGroup = [];
foreach ($allSettings as $setting) {
    $settingsByGroup[$setting['setting_group']][$setting['setting_key']] = $setting['setting_value'];
}

// SMS Status
echo "<h3>📱 SMS Configuration:</h3>";
$smsSettings = $settingsByGroup['sms_settings'] ?? [];
$smsEnabled = ($smsSettings['enable_sms'] ?? '0') === '1';
$smsConfigured = !empty($smsSettings['sms_api_key']) && !empty($smsSettings['sms_api_secret']) && !empty($smsSettings['sms_sender_id']);

echo "<ul>";
echo "<li><strong>Enabled:</strong> " . ($smsEnabled ? "✅ Yes" : "❌ No") . "</li>";
echo "<li><strong>Configured:</strong> " . ($smsConfigured ? "✅ Yes" : "❌ No") . "</li>";
if ($smsConfigured) {
    echo "<li><strong>Sender ID:</strong> " . htmlspecialchars($smsSettings['sms_sender_id']) . "</li>";
}
echo "</ul>";

// Email Status
echo "<h3>📧 Email Configuration:</h3>";
$emailSettings = $settingsByGroup['email_settings'] ?? [];
$emailEnabled = ($emailSettings['enable_email'] ?? '0') === '1';
$emailConfigured = !empty($emailSettings['smtp_host']) && !empty($emailSettings['smtp_port']) && 
                  !empty($emailSettings['smtp_username']) && !empty($emailSettings['smtp_password']);

echo "<ul>";
echo "<li><strong>Enabled:</strong> " . ($emailEnabled ? "✅ Yes" : "❌ No") . "</li>";
echo "<li><strong>Configured:</strong> " . ($emailConfigured ? "✅ Yes" : "❌ No") . "</li>";
if ($emailConfigured) {
    echo "<li><strong>SMTP Host:</strong> " . htmlspecialchars($emailSettings['smtp_host']) . "</li>";
    echo "<li><strong>SMTP Port:</strong> " . htmlspecialchars($emailSettings['smtp_port']) . "</li>";
}
echo "</ul>";

// WhatsApp Status
echo "<h3>💬 WhatsApp Configuration:</h3>";
$whatsappSettings = $settingsByGroup['whatsapp_settings'] ?? [];
$whatsappEnabled = ($whatsappSettings['enable_whatsapp'] ?? '0') === '1';

echo "<ul>";
echo "<li><strong>Enabled:</strong> " . ($whatsappEnabled ? "✅ Yes" : "❌ No") . "</li>";
echo "</ul>";
?>

<script>
function testReminder(customerId, type) {
    const resultsDiv = document.getElementById('results');
    resultsDiv.innerHTML = '<div class="info">🔄 Testing ' + type.toUpperCase() + ' reminder for customer ID ' + customerId + '...</div>';
    
    // Create test message
    const testMessage = `Dear Customer,

This is a TEST payment reminder from the POS system.

Please ignore this message - it's just a functionality test.

Test Type: ${type.toUpperCase()}
Test Time: ${new Date().toLocaleString()}

Thank you!
POS System`;

    // Make the same AJAX call that credit.php makes
    fetch('send_reminder.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({
            type: type,
            customer_id: customerId,
            message: testMessage
        })
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('Network response was not ok: ' + response.status);
        }
        return response.json();
    })
    .then(result => {
        let html = '<div class="' + (result.success ? 'success' : 'error') + '">';
        html += '<h3>🧪 Test Result for ' + type.toUpperCase() + ':</h3>';
        html += '<p><strong>Success:</strong> ' + (result.success ? 'Yes ✅' : 'No ❌') + '</p>';
        html += '<p><strong>Message:</strong> ' + escapeHtml(result.message) + '</p>';
        
        if (result.whatsapp_url) {
            html += '<p><strong>WhatsApp URL:</strong> <a href="' + escapeHtml(result.whatsapp_url) + '" target="_blank">Click to open WhatsApp</a></p>';
        }
        
        html += '<p><strong>Raw Response:</strong></p>';
        html += '<pre>' + JSON.stringify(result, null, 2) + '</pre>';
        html += '</div>';
        
        resultsDiv.innerHTML = html;
    })
    .catch(error => {
        console.error('Error:', error);
        resultsDiv.innerHTML = '<div class="error">' +
            '<h3>❌ Test Failed:</h3>' +
            '<p><strong>Error:</strong> ' + escapeHtml(error.message) + '</p>' +
            '<p>Check the browser console for more details.</p>' +
            '</div>';
    });
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Test if the send_reminder.php file exists
fetch('send_reminder.php', {
    method: 'HEAD'
})
.then(response => {
    if (!response.ok) {
        document.getElementById('results').innerHTML = '<div class="error">' +
            '❌ <strong>send_reminder.php file not found or not accessible!</strong><br>' +
            'This file is required for reminder functionality to work.' +
            '</div>';
    }
})
.catch(error => {
    document.getElementById('results').innerHTML = '<div class="error">' +
        '❌ <strong>Cannot access send_reminder.php:</strong> ' + error.message +
        '</div>';
});
</script>

<h2>🔍 How to Use This Tool:</h2>
<div class="info">
    <ol>
        <li><strong>Choose a customer:</strong> Pick a customer from the table above who has phone/email based on what you want to test</li>
        <li><strong>Click a test button:</strong> Try SMS (if customer has phone), Email (if customer has email), or WhatsApp (if customer has phone)</li>
        <li><strong>Check the results:</strong> The tool will show you exactly what happened</li>
        <li><strong>Verify delivery:</strong> If the test shows success, check if the actual SMS/Email/WhatsApp was received</li>
    </ol>
    
    <p><strong>Common Issues:</strong></p>
    <ul>
        <li><strong>Configuration not enabled:</strong> Go to Settings and enable SMS/Email/WhatsApp</li>
        <li><strong>Missing contact info:</strong> Customer needs phone number for SMS/WhatsApp, email for Email</li>
        <li><strong>Network errors:</strong> Check your internet connection and server setup</li>
        <li><strong>SMTP errors:</strong> Verify your email SMTP settings are correct</li>
    </ul>
</div>

</body>
</html> 